from quart import Quart, jsonify, request
from nebuloid.manifest import NebuloidManifest
import os

from .server import NebuloidServer
from nebuloid.orm import NebuloidORM
from nebuloid.api.expose_api import NebuloidExposeAPI
from nebuloid.plugins import NebuloidPluginManager

class Nebuloid:
    def __init__(self, profile=None):
        self.manifest = NebuloidManifest()
        self.orm = NebuloidORM(self.manifest)
        self.server = NebuloidServer(self.manifest, self.orm)
        self.expose_api = NebuloidExposeAPI(self.manifest)
        self.plugin_manager = NebuloidPluginManager(self.manifest, self.orm, self.expose_api)
        self.portal = self.expose_api.portal

        if profile:
            self.manifest.profile = profile

    @property
    def profile(self):
        return self.manifest.profile

    @profile.setter
    def profile(self, value):
        self.manifest.profile = value
    
    def init(self, manifest_file=None):
        if manifest_file:
            self.manifest.from_file(manifest_file)

        # Quart will now serve from these locations
        self.app = Quart(__name__, template_folder=None, static_folder=None)
        self.server.app = self.app

        self.server.mount()
        self.orm.mount()
        self.plugin_manager.mount()

    @property
    def wsgi(self):
        return self.app.wsgi_app  # Quart’s sync wrapper

    @property
    def asgi(self):
        return self.app  # native async app

    def run(self, host="127.0.0.1", port=5000, debug=True):
        self.expose_api.ready()
        self.server.ready()
        # Import only when running in dev mode
        from hypercorn.asyncio import serve
        from hypercorn.config import Config as HyperConfig
        import asyncio

        config = HyperConfig()
        config.bind = [f"{host}:{port}"]
        config.use_reloader = debug
        self.app.debug = debug

        async def main():
            # Run server + maintenance concurrently
            await asyncio.gather(
                serve(self.app, config),
                self.orm.maintenance_task(),
            )

        asyncio.run(main())